//
// Written by Aaron Skonnard (aaron@pluralsight.com)
// Blog: http://pluralsight.com/aaron/ 
//

using System;
using System.Configuration;
using System.IO;
using System.Text;
using System.Threading;
using System.Net;
using System.Xml;
using System.Web.Services.Description;

using Microsoft.Web.Services3;
using Microsoft.Web.Services3.Addressing;
using Microsoft.Web.Services3.Diagnostics;
using Microsoft.Web.Services3.Messaging;
using Microsoft.Web.Services3.Referral;

namespace Microsoft.Samples.HttpSys
{
    //
    // HttpSysAdapter provides a simple interface for receiving SoapEnvelope
    // messages via HTTP.SYS asynchronously. Simply instantiate the class, 
    // supplying the Uri you wish to listen on, and then call BeginReceive
    // providing a callback. Within your callback, call EndReceive to retrieve
    // the SoapEnvelope instance. You'll use this in your SoapTransport.
    //
	public class HttpSysAdapter : IDisposable
	{
        // Hold an instance of HttpListener for the lifetime of adapter
        private HttpListener listener;
        private bool listening;
        //
        // Initialize HttpListener using the supplied address (prefix)
        // and start listening
        //
        public HttpSysAdapter(Uri uri)
        {
            listening = false;
            listener = new HttpListener();
            listener.Prefixes.Add(uri.AbsoluteUri);
            listener.Start();
        }
        //
        // Call BeginReceive to begin listening via HTTP.SYS
        //
        public IAsyncResult BeginReceive(AsyncCallback callback)
		{
			this.listening = true;
            //
            // Simply create an instance of HttpSysAsyncResult, which will launch a 
            // thread to pull from HttpListener by calling GetContext
            //
			return new HttpSysAsyncResult(callback, this);
		}
        // 
        // Call EndReceive to retrieve to finish processing the received HTTP message
        // and to retrieve the SoapEnvelope (if one came in)
        //
		public SoapEnvelope EndReceive( IAsyncResult result )
		{
            // HttpSysAdapter supplied in IAsyncResult
            HttpSysAdapter.HttpSysAsyncResult ar =
                (HttpSysAdapter.HttpSysAsyncResult)result;

            if (null == ar)
                throw new ArgumentException("AsyncResult not obtained from HttpSysAdapter.BeginReceive()", "result");

            // Very important...causes exceptions thrown during async operation
            // to be rethrown here
            AsyncResult.End(result);

            // Check the HTTP method used in message, if GET, let's return the WSDL
            if (ar.Context.Request.HttpMethod.Equals("GET"))
            {
                try
                {
                    // See if we can retrieve the receiver for the supplied Url from SoapReceivers
                    EndpointReference er = new EndpointReference(ar.Context.Request.Url);
                    if (!SoapReceivers.Contains(er))
                        throw new Exception("Make sure you entered the exact same addressed supplied to SoapReceivers.Add, including the training forward slash '/'");
                    object obj = SoapReceivers.Receiver(er);

                    // If the target receiver is in a WsdlEnabledReceiver, call GetDescription and serialize WSDL
                    // into the response stream, and we're done
                    if (obj is WsdlEnabledReceiver)
                    {
                        WsdlEnabledReceiver receiver = obj as WsdlEnabledReceiver;
                        ServiceDescription desc = receiver.GetDescription(ar.Context.Request.Url);
                        using (XmlTextWriter tw = new XmlTextWriter(ar.Context.Response.OutputStream, Encoding.UTF8))
                        {
                            desc.ServiceDescriptions[0].Write(tw);
                        }
                    }
                    else
                        throw new Exception("In order to enable retrieving the service description (WSDL) via the browser, you must supply a Skonnard.Samples.HttpSys.AsmxReceiver when calling SoapReceivers.Add");
                }
                catch(Exception e)
                {
                    // Otherwise, we can't provide the WSDL so return some HTML telling the user what happened
                    using (XmlTextWriter tw = new XmlTextWriter(ar.Context.Response.OutputStream, Encoding.UTF8))
                    {
                        tw.WriteStartElement("html");
                        tw.WriteStartElement("body");
                        tw.WriteElementString("h1", "Service description unavailable");
                        tw.WriteElementString("p", e.Message);
                        tw.WriteEndElement();
                        tw.WriteEndElement();
                    }
                }
                // Returning a null SoapEnvelope prevents dispatching from continuing
                return null;
            }

            // If it wasn't a GET, we'll assume we need to deserialize the body into a SoapEnvelope
            // using the SoapPlainFormatter (we're not dealing with MTOM here)
            SoapEnvelope env = null;
            ISoapFormatter formatter = new SoapPlainFormatter();
            try
            {
                env = formatter.Deserialize(ar.Context.Request.InputStream);
            }
            catch
            {
                //non-soap messages shouldn't crash us - might want to do more error handling here
            }

            if (null != env)
            {
                // Setup WS-Addressing headers
                AddressingHeaders headers = env.Context.Addressing;
                Uri localEndpoint = ar.Context.Request.Url;
                Uri remoteEndpoint = new Uri(string.Format("http://{0}", ar.Context.Request.LocalEndPoint.ToString()));

                //This does most of the hard work for us
                headers.SetRequestHeaders(new Via(localEndpoint), new Via(remoteEndpoint));

                // save HttpListenerContext in request message - we'll need this later in the
                // HttpSysOutputChannel to transmit the response back down
                env.Context.SessionState.Set<HttpListenerContext>(ar.Context, "HttpSysAdapter");
            }

            return env;
		}

        //
        // Nested class that models async receive operation
        //
        private class HttpSysAsyncResult : AsyncResult
        {
            // Backpointer to HttpSysAdapter
            public HttpSysAdapter adapter;
            // Holds the retrieved HTTP message
            public HttpListenerContext Context;

            public HttpSysAsyncResult(AsyncCallback callback, object state)
                : base(callback, state)
            {
                adapter = state as HttpSysAdapter;
                // Start a thread to begin listening
                ThreadPool.QueueUserWorkItem(new WaitCallback(this.Receive), adapter);
            }

            private void Receive(object state)
            {
                // Retrieve the initialized HttpListener (held by adapter)
                HttpSysAdapter adapter = state as HttpSysAdapter;
                HttpListener listener = adapter.listener;

                bool caughtException = false;
                try
                {
                    // Call GetContext, which blocks, waiting for an HTTP
                    // message to arrive on the registered address prefix
                    Context = listener.GetContext();
                }
                // call base.Complete to signal we're done, this will cause
                // control to return to the supplied callback
                catch (Exception e)
                {
                    caughtException = true;
                    base.Complete(false, e);
                }
                if (!caughtException)
                    base.Complete(false);
            }
        }
        

        public bool Listening
        {
            get { return listening; }
        }


        #region IDisposable Members

        public void Dispose()
        {
            listener.Close();
        }

        #endregion
    }

}
