﻿using System;
using System.Collections.Generic;
using System.Reflection;
using System.Text;
using System.Threading;
using System.Web.Services.Protocols;

namespace ExceptionHelper
{
    /// <summary>
    /// Class that implements the methods needed to intercept the Invoke, EndInvoke,
    /// InvokeAsync methods on Web Service proxy classes generated by .NET/WSE. To 
    /// use this class, extend the partial web service proxy class and implement the
    /// Invoke, EndInvoke and InvokeAsync method and call this class' counterparts.
    /// </summary>
    public class ProxyExceptionHelper
    {
        private FaultExceptionMapperBase[] mExceptionMappers;

        public delegate object[] InvokeDelegate(string methodName, object[] parameters);
        public delegate object[] EndInvokeDelegate(IAsyncResult asyncResult);
        public delegate void InvokeAsyncDelegate(string methodName, object[] parameters,
            SendOrPostCallback callback, object userState);

        public ProxyExceptionHelper(params FaultExceptionMapperBase[] exceptionMappers)
        {
            mExceptionMappers = exceptionMappers;
        }

        /// <summary>
        /// Call the Invoke method given in the delegate and translate the exception
        /// (if any) if it is known. If the exception is not known, the exception is 
        /// simply rethrown.
        /// </summary>
        /// <param name="invoke">The Invoke delegate.</param>
        /// <param name="methodName">Name of the method.</param>
        /// <param name="parameters">The parameters.</param>
        /// <returns></returns>
        public object[] HandleInvoke(InvokeDelegate invoke,
            string methodName, object[] parameters)
        {
            try
            {
                return invoke(methodName, parameters);
            }
            catch (SoapException ex)
            {
                LookupExceptionAndThrowIfFound(ex);

                throw;
            }
        }

        /// <summary>
        /// Call the EndInvoke method given in the delegate and translate the exception
        /// (if any) if it is known. If the exception is not known, the exception is 
        /// simply rethrown.
        /// </summary>
        /// <param name="endInvoke">The EndInvoke delegate.</param>
        /// <param name="asyncResult">The async result.</param>
        /// <returns></returns>
        public object[] EndInvoke(EndInvokeDelegate endInvoke, IAsyncResult asyncResult)
        {
            try
            {
                return endInvoke(asyncResult);
            }
            catch (SoapException ex)
            {
                LookupExceptionAndThrowIfFound(ex);

                throw;
            }
        }

        /// <summary>
        /// Call the InvokeAsync method given in the delegate and translate the exception
        /// (if any) if it is known. If the exception is not known, the exception is 
        /// simply rethrown.
        /// </summary>
        /// <param name="invokeAsync">The InvokeAsync method.</param>
        /// <param name="methodName">Name of the method.</param>
        /// <param name="parameters">The parameters.</param>
        /// <param name="callback">The callback.</param>
        /// <param name="userState">User defined state.</param>
        public void InvokeAsync(InvokeAsyncDelegate invokeAsync,
            string methodName, object[] parameters,
            SendOrPostCallback callback, object userState)
        {
            invokeAsync(methodName, parameters,
                InvokeCallback,
                new object[] { callback, userState });
        }

        /// <summary>
        /// Callback method used by the InvokeAsync method. This method will try to translate
        /// an exception if one was thrown, and calls the "real" callback method that was 
        /// passed to the InvokeAsync method.
        /// </summary>
        /// <param name="a"></param>
        private void InvokeCallback(object a)
        {
            InvokeCompletedEventArgs invokeArgs = (InvokeCompletedEventArgs)a;
            object[] callbackArgs = (object[])invokeArgs.UserState;

            SoapException typedException = null;

            if (invokeArgs.Error != null && invokeArgs.Error is SoapException)
                typedException = LookupException((SoapException) invokeArgs.Error);

            InvokeCompletedEventArgs q = (InvokeCompletedEventArgs)
                Activator.CreateInstance(
                typeof(InvokeCompletedEventArgs),
                BindingFlags.NonPublic | BindingFlags.Instance | BindingFlags.CreateInstance,
                Type.DefaultBinder,
                new object[] 
                { 
                    invokeArgs.Results, 
                    typedException != null ? typedException : invokeArgs.Error, 
                    invokeArgs.Cancelled, 
                    callbackArgs[1] 
                },
                null);

            SendOrPostCallback realCallback = (SendOrPostCallback)callbackArgs[0];

            realCallback(q);
        }

        /// <summary>
        /// Tries to locate a known exception for the given SoapException, and if one
        /// is found, it is thrown.
        /// </summary>
        /// <param name="soapException">The SOAP exception.</param>
        private void LookupExceptionAndThrowIfFound(SoapException soapException)
        {
            SoapException typedException = LookupException(soapException);

            if (typedException != null)
                throw typedException;
        }

        /// <summary>
        /// Tries to locate a known exception for the given SoapExceptoin, if none is
        /// found, null is returned.
        /// </summary>
        /// <param name="soapException">The SOAP exception.</param>
        /// <returns></returns>
        private SoapException LookupException(SoapException soapException)
        {
            foreach (FaultExceptionMapperBase faultMapper in mExceptionMappers)
            {
                SoapException typedException = faultMapper.LookupAndCreateException(soapException);

                if (typedException != null)
                    return typedException;
            }

            return null;
        }
    }
}
