using System;
using System.Collections;
using System.Collections.Generic;
using System.Reflection;
using System.Xml;
using System.Xml.Serialization;

namespace GenerateFaultWrappersFromWsdl
{
	/// <summary>
	/// Import XML serializable types from an external assembly.
	/// </summary>
	class AssemblyTypeImporter
	{
        private Dictionary<XmlQualifiedName, string> mTypeMappings = new Dictionary<XmlQualifiedName, string>();

        /// <summary>
        /// Gets the type mappings.
        /// </summary>
        /// <value>The type mappings.</value>
        public Dictionary<XmlQualifiedName, string> TypeMappings
        {
            get
            {
                return mTypeMappings;
            }
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="T:AssemblyTypeImporter"/> class.
        /// </summary>
        /// <param name="typeMappings">The type mappings.</param>
        /// <param name="faultMappings">The fault mappings.</param>
        public AssemblyTypeImporter()
		{
            mTypeMappings = new Dictionary<XmlQualifiedName, string>();
		}

		/// <summary>
		/// Import the given assembly. This method will extract XML serializable types from
		/// the given assembly.
		/// </summary>
		/// <param name="assembly"></param>
		public void Import(Assembly assembly)
		{
			//
			// Look for XML serializable types
			//
			foreach(Type type in assembly.GetTypes())
			{
				// Look for the XmlTypeAttribute attribute
				object[] customAttributes = type.GetCustomAttributes(typeof(XmlTypeAttribute), true);

				if(customAttributes == null || customAttributes.Length == 0)
					continue;

				// Get the name and namespace of the Xml type
				XmlTypeAttribute typeAttribute = (XmlTypeAttribute) customAttributes[0];

                string xmlTypeName, xmlTypeNamespace;

                if (typeAttribute.AnonymousType)
                {
                    // This class is representing an anonymous type, i.e. one that is declared
                    // within an element, so the XmlRootAttribute will contain the name of the 
                    // element
                    customAttributes = type.GetCustomAttributes(typeof(XmlRootAttribute), true);

                    if (customAttributes == null || customAttributes.Length == 0)
                        continue;

                    XmlRootAttribute rootAttribute = (XmlRootAttribute)customAttributes[0];

                    xmlTypeName = rootAttribute.ElementName;
                    xmlTypeNamespace = rootAttribute.Namespace;
                }
                else
                {
                    xmlTypeName = typeAttribute.TypeName;
                    xmlTypeNamespace = typeAttribute.Namespace;

                }

                if (xmlTypeName == null || xmlTypeName == string.Empty)
                    // If not type name was set, we use the .NET class name as the type name
                    xmlTypeName = type.Name;

				mTypeMappings[new XmlQualifiedName(xmlTypeName, 
					xmlTypeNamespace)] = type.FullName;
			}
		}
	}
}
