//
// Written by Aaron Skonnard (aaron@pluralsight.com)
// Blog: http://pluralsight.com/aaron/ 
//

using System;
using System.Net;
using System.Collections;
using Microsoft.Web.Services3;
using Microsoft.Web.Services3.Addressing;
using Microsoft.Web.Services3.Diagnostics;
using Microsoft.Web.Services3.Messaging;

namespace Microsoft.Samples.HttpSys
{
	public class HttpSysTransport : SoapTransport, ISoapTransport
	{
        // Use to store instances of HttpSysAdapter
		private Hashtable httpSysAdapters;
        /// <summary>
        /// Added by Kristofer Agren, we need this in order to be able to use the
        /// default HTTP transport to handle WS-Anonynous URIs, i.e. receive the 
        /// resposne on the same connection as the request was sent on.
        /// </summary>
        private SoapHttpTransport mRegularHttpTransport;

		public HttpSysTransport()
		{
			httpSysAdapters = new Hashtable();
		}

		#region ISoapTransport Members

        //
        // Called by SoapReceivers when Add is called. We need to return an instance of
        // HttpSysInputChannel and begin listening for messages 
        // 
        // HttpSysAdapter is our class that manages dealing with HTTP.SYS
        //
		public ISoapInputChannel GetInputChannel(EndpointReference endpoint, SoapChannelCapabilities capabilities)
		{
			if( null == endpoint )
				throw new ArgumentNullException( "destination", "Cannot open input channel on null endpoint" );

            HttpSysInputChannel channel=null;

            // Lock the base class InputChannels collection
            lock (this.InputChannels.SyncRoot)
			{
                // Added by Kristofer Agren. Use the default HTTP transport for any inbound
                // channel that is identified by the WS-A anonymous role.
                if (endpoint.Address == WSAddressing.AnonymousRole)
                {
                    if (mRegularHttpTransport == null)
                        mRegularHttpTransport = new SoapHttpTransport();

                    return mRegularHttpTransport.GetInputChannel(endpoint, capabilities);
                }

                channel = new HttpSysInputChannel(endpoint);

                //
                // Create a new HttpSysAdapter based on the transport address, then
                // save it for future lookup
                //
                HttpSysAdapter adapter = FindOrCreateAdapter(endpoint.TransportAddress);
                //
                // Call BeginReceiver on HttpSysAdapter to begin listening for incoming
                // HTTP messages using HTTP.SYS
                //
                if (!adapter.Listening)
                    adapter.BeginReceive(new AsyncCallback(this.OnReceiveComplete));

                // Save the input channel is the base class InputChannels collection
				this.InputChannels.Add( channel );
			}
			return channel;
		}

        //
        // Called by SoapReceivers when a response is ready to be sent back to the client
        // 
        public ISoapOutputChannel GetOutputChannel(EndpointReference endpoint, SoapChannelCapabilities capabilities)
		{
            // Added by Kristofer Agren. Use the default HTTP transport for any outbound 
            // channel that is not using the WS-A anonymous role.
            lock (InputChannels.SyncRoot)
            {
                if (endpoint.Address != WSAddressing.AnonymousRole)
                {
                    if (mRegularHttpTransport == null)
                        mRegularHttpTransport = new SoapHttpTransport();

                    return mRegularHttpTransport.GetOutputChannel(endpoint, capabilities);
                }
            }

            // Make sure we can do what they want
			if( capabilities != SoapChannelCapabilities.None )
				throw new NotSupportedException( "Unsupported channel capabilities" );
			if( endpoint == null )
				throw new ArgumentNullException( "endpoint", "Cannot open output channel for null endpoint" );

            // Simply return an instance of HttpSysOutputChannel
			return new HttpSysOutputChannel( endpoint );
		}
        
        // 
        // This is used during WSDL generation to produce the right transport name
        //
        public string SoapBindingTransportUri
        {
            get { return "http://schemas.xmlsoap.org/soap/http"; }
        }
        
        #endregion

        //
        // This is called after the HttpSysAdapter receives an HTTP message
        //
		internal void OnReceiveComplete( IAsyncResult result )
		{
            // HttpSysAdapter instance is supplied in the IAsyncResult
			HttpSysAdapter adapter = result.AsyncState as HttpSysAdapter;
			SoapEnvelope envelope;

			try
			{
                // Retrieve the SoapEnvelope from the HttpSysAdapter
				envelope = adapter.EndReceive( result );
                // Start the adapter listening again (or it will stop after 1 message)
                adapter.BeginReceive(new AsyncCallback(this.OnReceiveComplete));

			}
			catch( AsynchronousOperationException e )
			{
				// Any exceptions that occured on the async thread will be caught here
				EventLog.WriteError( "HttySysTransport failure: " + e.InnerException.Message );
				adapter.BeginReceive( new AsyncCallback( this.OnReceiveComplete ) );
                return;
			}

            //This conditional was obtained by looking at the implementation of the SoapTcpTransport
            if ( ( ((envelope != null) && !base.DispatchMessage(envelope)) && (envelope.Fault == null)) &&
                 ( (envelope.Context.Addressing.RelatesTo == null) || 
                   (envelope.Context.Addressing.RelatesTo.RelationshipType != WSAddressing.AttributeValues.Reply)))
            {
                //If we get here, we have an undeliverable message. We need to send a fault back to the originator.
                try
                {
                    SoapEnvelope faultMessage = base.GetFaultMessage(envelope,
                        new AddressingFault(AddressingFault.DestinationUnreachableMessage,
                        AddressingFault.DestinationUnreachableCode));

                    // copy HttpListenerContext to fault message so Send works on HttpSysOutputChannel
                    faultMessage.Context.SessionState.Set<HttpListenerContext>(
                        envelope.Context.SessionState.Get<HttpListenerContext>("HttpSysAdapter"), "HttpSysAdapter");

                    //We can't send faults back to the anonymous role
                    if (!faultMessage.Context.Addressing.Destination.TransportAddress.Equals(WSAddressing.AnonymousRole))
                    {
                        ISoapOutputChannel outputChannel = SoapTransport.StaticGetOutputChannel(faultMessage.Context.Addressing.Destination);
                        outputChannel.Send(faultMessage);
                    }
                }
                catch (Exception e)
                {
                    EventLog.WriteError(String.Format("WSE HttpSysTransport failure: {0}", e));
                }
            }
		}

        // 
        // The first time a Uri is used, creates an HttpSysAdapter and saves 
        // the reference in a hashtable using the Uri as the key
        //
        // Future calls for a particular Uri, it simply returns the stored
        // HttpSysAdapter
        //
		internal HttpSysAdapter FindOrCreateAdapter( Uri uri )
		{
			lock( this.httpSysAdapters )
			{
				if( !httpSysAdapters.Contains( uri ) )
				{
					HttpSysAdapter adapter = new HttpSysAdapter( uri );
					httpSysAdapters.Add( uri, adapter);
				}
				
				return (HttpSysAdapter) httpSysAdapters[ uri ];
			}
		}
    }
}
