﻿using System;
using System.Collections.Generic;
using System.CodeDom;
using System.CodeDom.Compiler;
using System.IO;
using System.Text;
using System.Web.Services.Description;
using System.Xml;

using Microsoft.CSharp;
using Microsoft.VisualBasic;

namespace GenerateFaultWrappersFromWsdl
{
    class Program
    {
        static void Main(string[] args)
        {
            //
            // Read all the command-line options
            //
            List<string> wsdlFiles = new List<string>();
            List<string> existingTypeFiles = new List<string>();
            
            string ns = null, outputFile = null;
            string language = "CS";
            bool verbose = false;
            bool generateExceptionMapperClass = false;

			for(int i = 0; i < args.Length; i++)
			{
				if(args[i][0] == '/' || args[i][0] == '-')
				{
					string option = args[i].Substring(1);
                    string val = "";

                    if(i + 1 < args.Length)
					    val = args[i + 1]; 

					switch(option)
					{
                        case "v":
                            verbose = true;
                            break;

                        case "g":
                            generateExceptionMapperClass = true;
                            break;
                        
                        case "i":
							wsdlFiles.Add(val);
							break;

                        case "ext":
                            existingTypeFiles.Add(val);
                            break;

						case "ns":
							ns = val;
							break;

						case "o":
							outputFile = val;
							break;

                        case "l":
                            language = val;
                            break;
					}
				}
			}

			if(wsdlFiles.Count == 0 || outputFile == null)
			{
                // At least one wsdlFile must be provided as well as an output file
				PrintUsage();
				return;
			}

            try
            {
                //
                // Import types from the given list of source files and/or assemblies
                //
                Dictionary<XmlQualifiedName, string> existingTypeMappings =
                    ImportTypes(existingTypeFiles.ToArray());

                //
                // Load all WSDLs
                //
                ServiceDescriptionCollection sdc = new ServiceDescriptionCollection();

                foreach (string wsdlFile in wsdlFiles)
                    sdc.Add(ServiceDescription.Read(wsdlFile));

                //
                // Generate the fault wrapper classes
                //
                CodeNamespace cn = new CodeNamespace(ns);

                FaultGenerator faultGenerator = new FaultGenerator(sdc, cn, existingTypeMappings);

                faultGenerator.GenerateExceptionClasses();

                if(generateExceptionMapperClass)
                    faultGenerator.GenerateExceptionMapper();

                //
                // Write the output file
                //
                CodeGeneratorOptions codeGeneratorOptions = new CodeGeneratorOptions();

                CodeDomProvider codeDomProvider;

                // Get a CodeDOM
                switch (language)
                {
                    case "CS":
                        codeDomProvider = new Microsoft.CSharp.CSharpCodeProvider();
                        break;

                    case "VB":
                        codeDomProvider = new Microsoft.VisualBasic.VBCodeProvider();
                        break;

                    default:
                        // Try using the CodeDomProvider.CreateProvider implementation
                        codeDomProvider = CodeDomProvider.CreateProvider(language);

                        if(codeDomProvider == null)
                            throw new ApplicationException(string.Format(
                                "The language type {0} is not known", language));

                        break;
                }

                using (StreamWriter sw = new StreamWriter(outputFile))
                    codeDomProvider.GenerateCodeFromNamespace(cn, sw, codeGeneratorOptions);

            }
            catch (Exception ex)
            {
                if (verbose)
                    Console.WriteLine(ex.ToString());
                else
                {
                    Console.WriteLine("Failed to generate classes: " + ex.Message);
                    Console.WriteLine();
                    Console.WriteLine("To get more detailed error information use the /v " +
                        "command-line switch");
                }
                
                Console.WriteLine();
            }
                        
        }

        /// <summary>
        /// Prints the usage.
        /// </summary>
        private static void PrintUsage()
        {
            Console.WriteLine("Usage: GenerateFaultWrappersFromWsdl /i <wsdlfile> /o <outputfile>" + 
                "/ext <existingtypefile> /ns <namespace of generated classes> /l <language> /g /v");

            Console.WriteLine();
            Console.WriteLine("/i <wsdlfile>");
            Console.WriteLine("    Import the given WSDL file. /i can be supplied multiple times. At least one WSDL file MUST be supplied.");
            Console.WriteLine();
            Console.WriteLine("/o <outputfile>");
            Console.WriteLine("    Output source file, e.g. MyFaults.cs or MyFaults.vb. The outputfile MUST be supplied.");
            Console.WriteLine();
            Console.WriteLine("/ext <existingtypefile>");
            Console.WriteLine("    A source file (C# or VB.NET) or an assembly that contain the .NET classes.");
            Console.WriteLine();
            Console.WriteLine("/ns <existingtypefile>");
            Console.WriteLine("    The namespace used for the generated classes and the optional exception mapper class.");
            Console.WriteLine();
            Console.WriteLine("/l <language>");
            Console.WriteLine("    The language to generate the source file in, available values are CS or VB or any string that CodeDomProvider.CreateProvider accepts. Default is CS.");
            Console.WriteLine();
            Console.WriteLine("/g");
            Console.WriteLine("    Generate an exception mapper class. This is only required for client applications.");
            Console.WriteLine();
            Console.WriteLine("/v");
            Console.WriteLine("    Use verbose error reporting.");
            Console.WriteLine();
            Console.WriteLine("Example:");
            Console.WriteLine();
            Console.WriteLine("GenerateFaultWrappersFromWsdl /i MyService1.wsdl /i MyService2.wsdl " +
                "/o FaultWrappers.vb /ext FaultTypes.cs /ext SomeMoreFaultTypes.dll /ns MyNamespace /l VB");
            Console.WriteLine();
        }

        /// <summary>
        /// Import all the existing XML-serializable types from the given list of 
        /// references, which can either be a C# source file (.cs) or an assembly (.dll).
        /// </summary>
        /// <param name="existingTypeFiles"></param>
        /// <returns></returns>
        private static Dictionary<XmlQualifiedName, string> ImportTypes(
            string[] existingTypeFiles)
        {
            AssemblyTypeImporter assemblyTypeImporter = new AssemblyTypeImporter();

            // First get all the source files so that we can compile them into an assembly
            Dictionary<string, List<string>> sourceFilesByLanguage = new Dictionary<string, List<string>>();
            List<string> assemblyFiles = new List<string>();

            foreach (string existingTypeFile in existingTypeFiles)
            {
                string extension = Path.GetExtension(existingTypeFile);

                if (extension == ".dll")
                    assemblyFiles.Add(existingTypeFile);

                else if(CodeDomProvider.IsDefinedExtension(extension))
                {
                    string languageFromExtension = CodeDomProvider.GetLanguageFromExtension(extension);
                    List<string> sourceFiles;
                    
                    if(!sourceFilesByLanguage.TryGetValue(languageFromExtension, out sourceFiles))
                    {
                        sourceFiles = new List<string>();
                        sourceFilesByLanguage[languageFromExtension] = sourceFiles;
                    }

                    sourceFiles.Add(existingTypeFile);
                }
                else
                    throw new ApplicationException(string.Format(
                        "Unknown file extension " + Path.GetExtension(existingTypeFile)));
            }

            // Compile all source files and import the types from the resulting assembly
            if (sourceFilesByLanguage.Count > 0)
            {
                // Populate a list of all references
                List<string> references = new List<string>();

                references.Add("System.dll");
                references.Add("System.Web.dll");
                references.Add("System.Web.Services.dll");
                references.Add("System.Xml.dll");
                references.Add(typeof(Microsoft.Web.Services3.WseProtocolFactory).Assembly.Location);

                // Add all references assemblies to the list as well
                foreach (string assemblyName in assemblyFiles)
                    references.Add(assemblyName);

                CompilerParameters compilerParameters = new CompilerParameters(
                    references.ToArray());

                compilerParameters.GenerateInMemory = true;

                //
                // Compile all the source files
                //
                foreach (KeyValuePair<string, List<string>> keyValuePair in sourceFilesByLanguage)
                {
                    List<string> sourceFiles = keyValuePair.Value;

                    CodeDomProvider provider = CodeDomProvider.CreateProvider(keyValuePair.Key);

                    CompilerResults compilerResults = provider.CompileAssemblyFromFile(
                        compilerParameters, sourceFiles.ToArray());

                    if (compilerResults.NativeCompilerReturnValue != 0)
                    {
                        // Failed to compile
                        StringBuilder sb = new StringBuilder();

                        foreach (string s in compilerResults.Output)
                            sb.Append(s);

                        throw new ApplicationException(string.Format(
                            "Unable to compile reference files: {0}",
                            sb.ToString()));
                    }

                    // Import the assembly
                    assemblyTypeImporter.Import(compilerResults.CompiledAssembly);
                }
            }


            // Process assemblies
            foreach (string strAssemblyFileName in assemblyFiles)
            {
                // Load the assembly
                System.Reflection.AssemblyName assemblyName =
                    System.Reflection.AssemblyName.GetAssemblyName(strAssemblyFileName);

                assemblyTypeImporter.Import(System.Reflection.Assembly.Load(assemblyName));
            }

            return assemblyTypeImporter.TypeMappings;
        }
    }
}
