﻿using System;
using System.Collections.Generic;
using System.CodeDom;
using System.Text;
using System.Web.Services.Description;
using System.Web.Services.Protocols;
using System.Xml;
using System.Xml.Schema;
using System.Xml.Serialization;

namespace GenerateFaultWrappersFromWsdl
{
    /// <summary>
    /// Class that generates wrapper classes for operations with structured faults, i.e.
    /// faults that have a message, with one part and that part refers to an XML Schema 
    /// element.
    /// </summary>
    class FaultGenerator
    {
        private ServiceDescriptionCollection mServiceDescriptionCollection;
        private CodeNamespace mCodeNamespace;
        /// <summary>
        /// Dictionary of all the existing schema element-to-generated class mappings.
        /// </summary>
        private Dictionary<XmlQualifiedName, string> mExistingTypeMappings = 
            new Dictionary<XmlQualifiedName, string>();
        /// <summary>
        /// All the fault types that have been created.
        /// </summary>
        private Dictionary<XmlQualifiedName, string> mFaultMappings =
            new Dictionary<XmlQualifiedName, string>();

        /// <summary>
        /// Initializes a new instance of the <see cref="T:FaultGenerator"/> class.
        /// </summary>
        /// <param name="sdc">The SDC.</param>
        /// <param name="cn">The cn.</param>
        /// <param name="existingTypeMappings">The existing type mappings.</param>
        public FaultGenerator(ServiceDescriptionCollection sdc, CodeNamespace cn,
            Dictionary<XmlQualifiedName, string> existingTypeMappings)
        {
            mCodeNamespace = cn;
            mServiceDescriptionCollection = sdc;
            mExistingTypeMappings = existingTypeMappings;
        }

        /// <summary>
        /// Generates exception classes for all fault types found in the collection of WSDLs
        /// that was passed to the constructor.
        /// </summary>
        public void GenerateExceptionClasses()
        {
            //
            // Loop through all the port types and look for operations with
            // fault elements that bind to messages with elements
            //
            foreach(ServiceDescription sdc in mServiceDescriptionCollection)
            {
                foreach (PortType portType in sdc.PortTypes)
                {
                    foreach (Operation operation in portType.Operations)
                    {
                        foreach (OperationMessage operationFault in operation.Faults)
                        {
                            // Lookup message and check if the message has one part and that
                            // part is an element 
                            Message faultMessage = mServiceDescriptionCollection.GetMessage(
                                operationFault.Message);

                            if (faultMessage == null)
                                throw new ApplicationException(string.Format(
                                    "Unable to locate message named {0}:{1}",
                                    operationFault.Message.Namespace,
                                    operationFault.Message.Name));

                            if (faultMessage.Parts.Count != 1 ||
                                faultMessage.Parts[0].Element.IsEmpty)
                                // Fault elements that have more than one part or not an element
                                // type are not valid
                                continue;

                            // Lookup the element
                            MessagePart messagePart = faultMessage.Parts[0];
                            XmlQualifiedName elementName = messagePart.Element;

                            // Emit the code for this element
                            GenerateSoapException(elementName);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Generate a class that maps QNames to exception classes
        /// </summary>
        public void GenerateExceptionMapper()
        {
            CodeTypeDeclaration typeDeclaration = new CodeTypeDeclaration("FaultExceptionMapper");

            typeDeclaration.BaseTypes.Add(new CodeTypeReference("ExceptionHelper.FaultExceptionMapperBase"));

            //
            // Emit the "LookupAndCreateException" method that will return an instance
            // of the typed exception, if a match was found.
            //
            CodeMemberMethod lookupMethod = new CodeMemberMethod();

            lookupMethod.Name = "LookupAndCreateException";
            lookupMethod.Attributes = MemberAttributes.Public | MemberAttributes.Override;
            lookupMethod.ReturnType = new CodeTypeReference(typeof(SoapException));

            lookupMethod.Parameters.Add(
                new CodeParameterDeclarationExpression(
                new CodeTypeReference(typeof(SoapException)), 
                "ex"));

            // XmlQualifiedName qualifiedDetailName;
            lookupMethod.Statements.Add(
                new CodeVariableDeclarationStatement(
                new CodeTypeReference(typeof(XmlQualifiedName)), "qualifiedDetailName"));

            // if(ex.Detail != null && ex.Detail.HasChildNodes)
            //      qualifiedDetailName = new XmlQualifiedName(ex.Detail.FirstChild.LocalName, 
            //          ex.Detail.FirstChild.NamespaceURI);
            // else
            //      return null;
            lookupMethod.Statements.Add(
                new CodeConditionStatement(
                new CodeBinaryOperatorExpression(
                new CodeBinaryOperatorExpression(
                new CodePropertyReferenceExpression(
                new CodeVariableReferenceExpression("ex"), "Detail"),
                CodeBinaryOperatorType.IdentityInequality,
                new CodePrimitiveExpression(null)), 
                CodeBinaryOperatorType.BooleanAnd,

                new CodePropertyReferenceExpression(
                new CodePropertyReferenceExpression(
                new CodeVariableReferenceExpression("ex"), "Detail"), "HasChildNodes")),

                new CodeStatement[]
                {
                    new CodeAssignStatement(
                    new CodeVariableReferenceExpression("qualifiedDetailName"),
                    new CodeObjectCreateExpression(
                    new CodeTypeReference(typeof(XmlQualifiedName)),
                    new CodePropertyReferenceExpression(
                    new CodePropertyReferenceExpression(
                    new CodePropertyReferenceExpression(
                    new CodeVariableReferenceExpression("ex"), "Detail"), "FirstChild"), "LocalName"),
                    new CodePropertyReferenceExpression(
                    new CodePropertyReferenceExpression(
                    new CodePropertyReferenceExpression(
                    new CodeVariableReferenceExpression("ex"), "Detail"), "FirstChild"), "NamespaceURI")))
                },
                new CodeStatement[] 
                { 
                    new CodeMethodReturnStatement(new CodePrimitiveExpression(null))
                }));

            foreach (KeyValuePair<XmlQualifiedName, string> keyValuePair in mFaultMappings)
            {
                XmlQualifiedName qname = keyValuePair.Key;

                // if(qualifiedDetailName.Name == thename && qualifiedDetailName.NamespaceURI = thens)
                //      return new theexceptiontype;
                lookupMethod.Statements.Add(
                    new CodeConditionStatement(
                    new CodeBinaryOperatorExpression(
                    new CodeBinaryOperatorExpression(
                    new CodePropertyReferenceExpression(new CodeVariableReferenceExpression("qualifiedDetailName"), "Namespace"),
                    CodeBinaryOperatorType.ValueEquality,
                    new CodePrimitiveExpression(qname.Namespace)),
                    CodeBinaryOperatorType.BooleanAnd,
                    new CodeBinaryOperatorExpression(
                    new CodePropertyReferenceExpression(
                    new CodeVariableReferenceExpression("qualifiedDetailName"), "Name"),
                    CodeBinaryOperatorType.ValueEquality,
                    new CodePrimitiveExpression(qname.Name))),
                    new CodeMethodReturnStatement(
                    new CodeObjectCreateExpression(
                    new CodeTypeReference(keyValuePair.Value),
                    new CodePropertyReferenceExpression(
                    new CodeVariableReferenceExpression("ex"), "Message"),
                    new CodePropertyReferenceExpression(
                    new CodeVariableReferenceExpression("ex"), "Code"),
                    new CodePropertyReferenceExpression(
                    new CodeVariableReferenceExpression("ex"), "Actor"),
                    new CodePropertyReferenceExpression(
                    new CodeVariableReferenceExpression("ex"), "Detail")
                    ))));
            }

            // return null;
            lookupMethod.Statements.Add(
                new CodeMethodReturnStatement(
                new CodePrimitiveExpression(null)));

            typeDeclaration.Members.Add(lookupMethod);

            mCodeNamespace.Types.Add(typeDeclaration);
        }

        /// <summary>
        /// Generate the class that derives from SoapException and that wraps the 
        /// fault with detail described by the XML Schema element identified by the
        /// given QName.
        /// </summary>
        /// <param name="elementName"></param>
        private void GenerateSoapException(XmlQualifiedName elementName)
        {
            if (mFaultMappings.ContainsKey(elementName))
                // Fault has already been generated
                return;

            //
            // Format the class name
            //
            string className = elementName.Name + "Exception";

            // Make sure the first letter is in upper case
            if (char.IsLower(className[0]))
                className = char.ToUpper(className[0]) + className.Substring(1);

            //
            string xmlSerializableType = GetNameOfGeneratedTypeForSchemaElement(elementName);

            //
            // Emit the type
            //
            CodeTypeDeclaration typeDeclaration = new CodeTypeDeclaration(className);

            //
            // Inherit from SoapException
            //
            typeDeclaration.BaseTypes.Add(typeof(SoapException));

            //
            // Emit the private constructor
            //
            /*
            CodeConstructor privateConstructor = new CodeConstructor();

            privateConstructor.Attributes = MemberAttributes.Private;

            privateConstructor.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(typeof(string)), "message"));

            privateConstructor.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(typeof(XmlQualifiedName)), "code"));

            privateConstructor.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(typeof(string)), "actor"));

            privateConstructor.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(typeof(XmlNode)), "detail"));

            privateConstructor.BaseConstructorArgs.Add(
                new CodeVariableReferenceExpression("message"));
            privateConstructor.BaseConstructorArgs.Add(
                new CodeVariableReferenceExpression("code"));
            privateConstructor.BaseConstructorArgs.Add(
                new CodeVariableReferenceExpression("actor"));
            privateConstructor.BaseConstructorArgs.Add(
                new CodeVariableReferenceExpression("detail"));

            typeDeclaration.Members.Add(privateConstructor);
             * 
             */

            //
            // Emit the public constructors
            //

            // All arguments, detail as xml serializable class
            CodeConstructor publicConstructorFull = new CodeConstructor();

            publicConstructorFull.Attributes = MemberAttributes.Public;

            publicConstructorFull.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(typeof(string)), "message"));

            publicConstructorFull.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(typeof(XmlQualifiedName)), "code"));

            publicConstructorFull.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(typeof(string)), "actor"));

            publicConstructorFull.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(xmlSerializableType), "detail"));

            publicConstructorFull.BaseConstructorArgs.Add(
                new CodeVariableReferenceExpression("message"));

            publicConstructorFull.BaseConstructorArgs.Add(
                new CodeVariableReferenceExpression("code"));

            publicConstructorFull.BaseConstructorArgs.Add(
                new CodeVariableReferenceExpression("actor"));

            publicConstructorFull.BaseConstructorArgs.Add(
                new CodeMethodInvokeExpression(
                new CodeMethodReferenceExpression(
                new CodeTypeReferenceExpression(
                new CodeTypeReference(className)),
                "SerializeTypeToXml"),
                new CodeVariableReferenceExpression("detail")));

            typeDeclaration.Members.Add(publicConstructorFull);

            // All arguments, detail as XML
            CodeConstructor publicConstructorFullXml = new CodeConstructor();

            publicConstructorFullXml.Attributes = MemberAttributes.Public;

            publicConstructorFullXml.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(typeof(string)), "message"));

            publicConstructorFullXml.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(typeof(XmlQualifiedName)), "code"));

            publicConstructorFullXml.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(typeof(string)), "actor"));

            publicConstructorFullXml.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(typeof(XmlNode)), "detail"));

            publicConstructorFullXml.BaseConstructorArgs.Add(
                new CodeVariableReferenceExpression("message"));

            publicConstructorFullXml.BaseConstructorArgs.Add(
                new CodeVariableReferenceExpression("code"));

            publicConstructorFullXml.BaseConstructorArgs.Add(
                new CodeVariableReferenceExpression("actor"));

            publicConstructorFullXml.BaseConstructorArgs.Add(
                new CodeVariableReferenceExpression("detail"));

            typeDeclaration.Members.Add(publicConstructorFullXml);

            // Minimum arguments
            CodeConstructor publicConstructorMinimum = new CodeConstructor();

            publicConstructorMinimum.Attributes = MemberAttributes.Public;

            publicConstructorMinimum.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(typeof(XmlQualifiedName)), "code"));

            publicConstructorMinimum.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(xmlSerializableType), "detail"));

            publicConstructorMinimum.ChainedConstructorArgs.Add(
                new CodePrimitiveExpression("An exception occurred"));
            
            publicConstructorMinimum.ChainedConstructorArgs.Add(
                new CodeVariableReferenceExpression("code"));
            
            publicConstructorMinimum.ChainedConstructorArgs.Add(
                new CodePrimitiveExpression(""));

            publicConstructorMinimum.ChainedConstructorArgs.Add(
                new CodeVariableReferenceExpression("detail"));

            typeDeclaration.Members.Add(publicConstructorMinimum);

            //
            // Emit the SerializeTypeToXml method
            //


            // public static XmlElement SerializeTypeToXml(TheType detail) { .. }
            CodeMemberMethod createMethod = new CodeMemberMethod();
            createMethod.Name = "SerializeTypeToXml";
            createMethod.Attributes = MemberAttributes.Static |
                MemberAttributes.Public;
            createMethod.ReturnType = new CodeTypeReference(typeof(XmlElement));

            createMethod.Parameters.Add(new CodeParameterDeclarationExpression(
                new CodeTypeReference(xmlSerializableType), "detail"));

            typeDeclaration.Members.Add(createMethod);

            // XmlSerializer xmlSerializer = new XmlSerializer(type)
            createMethod.Statements.Add(
                new CodeVariableDeclarationStatement(
                typeof(XmlSerializer), "xmlSerializer",
                new CodeObjectCreateExpression(
                typeof(XmlSerializer), new CodeExpression[]
				{
					new CodeTypeOfExpression(xmlSerializableType)
				})));

            // MemoryStream ms = new MemoryStrem();
            createMethod.Statements.Add(
                new CodeVariableDeclarationStatement(
                typeof(System.IO.MemoryStream), "ms",
                new CodeObjectCreateExpression(
                typeof(System.IO.MemoryStream),
                new CodeExpression[0])));

            // xmlSerializer.Serialize(ms, arg);
            createMethod.Statements.Add(
                new CodeMethodInvokeExpression(
                new CodeVariableReferenceExpression("xmlSerializer"),
                "Serialize", new CodeExpression[]
				{
					new CodeVariableReferenceExpression("ms"),
					new CodeVariableReferenceExpression("detail")
				}));

            // ms.Seek(0, System.IO.SeekOrigin.Begin);
            createMethod.Statements.Add(
                new CodeMethodInvokeExpression(
                new CodeVariableReferenceExpression("ms"), "Seek",
                new CodeExpression[]
				{
					new CodePrimitiveExpression(0),
					new CodeFieldReferenceExpression(
					new CodeTypeReferenceExpression("System.IO.SeekOrigin"), "Begin")
				}));

            // XmlDocument xmlDoc = new XmlDocument();
            createMethod.Statements.Add(
                new CodeVariableDeclarationStatement(
                typeof(XmlDocument), "xmlDoc",
                new CodeObjectCreateExpression(typeof(XmlDocument),
                new CodeExpression[0])));

            // xmlDoc.Load(ms);
            createMethod.Statements.Add(
                new CodeMethodInvokeExpression(
                new CodeVariableReferenceExpression("xmlDoc"),
                "Load", new CodeExpression[] 
				{
					new CodeVariableReferenceExpression("ms")
				}));

            // XmlElement rootDetail = doc.CreateNode(XmlNodeType.Element, 
            //      SoapException.DetailElementName.Name, SoapException.DetailElementName.Namespace);
            createMethod.Statements.Add(
                new CodeVariableDeclarationStatement(
                new CodeTypeReference(typeof(XmlElement)), "rootDetail",
                new CodeCastExpression(
                new CodeTypeReference(typeof(XmlElement)),
                new CodeMethodInvokeExpression(
                new CodeMethodReferenceExpression(
                new CodeVariableReferenceExpression("xmlDoc"), "CreateNode"),
                new CodeFieldReferenceExpression(
                new CodeTypeReferenceExpression(typeof(XmlNodeType)), "Element"),
                new CodePropertyReferenceExpression(
                new CodeFieldReferenceExpression(
                new CodeTypeReferenceExpression(typeof(SoapException)), "DetailElementName"), "Name"),
                new CodePropertyReferenceExpression(
                new CodeFieldReferenceExpression(
                new CodeTypeReferenceExpression(typeof(SoapException)), "DetailElementName"), "Namespace")
            ))));

            // rootDetail.AppendChild(doc.RemoveChild(doc.DocumentElement));
            createMethod.Statements.Add(
                new CodeMethodInvokeExpression(
                new CodeMethodReferenceExpression(
                new CodeVariableReferenceExpression("rootDetail"), "AppendChild"),
                new CodeMethodInvokeExpression(
                new CodeMethodReferenceExpression(
                new CodeVariableReferenceExpression("xmlDoc"), "RemoveChild"),
                new CodePropertyReferenceExpression(
                new CodeVariableReferenceExpression("xmlDoc"),
                "DocumentElement"))));

            // return rootDetail;
            createMethod.Statements.Add(
                new CodeMethodReturnStatement(
                new CodeVariableReferenceExpression("rootDetail")));

            //
            // Add the TypedDetail property
            //
            CodeMemberProperty prop = new CodeMemberProperty();
            prop.Name = "TypedDetail";
            prop.Type = new CodeTypeReference(xmlSerializableType);
            prop.Attributes = MemberAttributes.Public;

            // XmlElement detail = (XmlElement) Detail.FirstChild;
            prop.GetStatements.Add(
                new CodeVariableDeclarationStatement(
                new CodeTypeReference(typeof(XmlElement)), "detail",
                new CodeCastExpression(
                new CodeTypeReference(typeof(XmlElement)),
                new CodePropertyReferenceExpression(
                new CodePropertyReferenceExpression(
                new CodeBaseReferenceExpression(), "Detail"), "FirstChild"))));


            //XmlSerializer xmlSerializer = new XmlSerializer(type);
            prop.GetStatements.Add(new CodeVariableDeclarationStatement(
                new CodeTypeReference(typeof(XmlSerializer)), "xmlSerializer",
                new CodeObjectCreateExpression(
                new CodeTypeReference(typeof(XmlSerializer)),
                new CodeTypeOfExpression(xmlSerializableType))));

            //return (thetype) xmlSerializer.Deserialize(new XmlNodeReader(detail));
            prop.GetStatements.Add(new CodeMethodReturnStatement(
                new CodeCastExpression(
                new CodeTypeReference(xmlSerializableType),
                new CodeMethodInvokeExpression(
                new CodeMethodReferenceExpression(
                new CodeVariableReferenceExpression("xmlSerializer"), "Deserialize"),
                new CodeObjectCreateExpression(
                new CodeTypeReference(typeof(XmlNodeReader)),
                new CodeVariableReferenceExpression("detail"))))));

            typeDeclaration.Members.Add(prop);

            mCodeNamespace.Types.Add(typeDeclaration);

            mFaultMappings[elementName] = typeDeclaration.Name;
           
        }

        /// <summary>
        /// Get the .NET type name for the class that represents the XML Schema element 
        /// identified with the given QName.
        /// </summary>
        /// <param name="elementName"></param>
        /// <returns></returns>
        private string GetNameOfGeneratedTypeForSchemaElement(XmlQualifiedName elementName)
        {
            string typeName;
            
            if(!mExistingTypeMappings.TryGetValue(elementName, out typeName))
                throw new ApplicationException(string.Format(
                    "Unable to find an existing XML serializable type for the " +
                    "element named {0}:{1}", elementName.Namespace, elementName.Name));

            return typeName;
        }
    }
}
